﻿using System;
using System.Collections.Generic;
using System.Text;
using System.IO;
using System.Windows.Forms;
using KeyConfig.Config;

namespace KeyConfig.Config
{
	internal class ConfigWriter : IDisposable
	{
		//----------// //----------// //----------// //----------//
		//
		//	Constructor
		//
		//----------// //----------// //----------// //----------//

		public ConfigWriter(Stream input)
		{
			if (input == null) throw new ArgumentNullException("input");

			this.sw = new StreamWriter(input, Encoding.GetEncoding("shift-jis"));
		}

		//----------// //----------// //----------// //----------//
		//
		//	Fields
		//
		//----------// //----------// //----------// //----------//

		StreamWriter sw;

		//----------// //----------// //----------// //----------//
		//
		//	Methods
		//
		//----------// //----------// //----------// //----------//

		internal void Write(ConfigData config)
		{
			if (config == null) throw new ArgumentNullException("config");

			WriteKeySection(config.KeyList);

			WriteSeparator();

			WritePadSection(config.PadList);

			WriteSeparator();

			WriteScreenSection(config.FullScreen, config.SyncTimer, config.SkipFrame);
		}

		internal void Close()
		{
			sw.Flush();
			sw.Close();
		}

		public void Dispose()
		{
			Close();
		}

		//----------// //----------// //----------// //----------//
		//
		//	Methods (private)
		//
		//----------// //----------// //----------// //----------//

		private void WriteKeySection(List<KeyValuePair<string, int>> keyList)
		{
			WriteComment(SectionComment.Key);
			WriteSectionName(SectionName.Key);

			for (int i = 0; i < keyList.Count; i++)
			{
				string line = String.Format("{0}={1}", keyList[i].Key, keyList[i].Value.ToString());
				sw.WriteLine(line);
			}
		}

		private void WritePadSection(List<KeyValuePair<string, int>> padList)
		{
			WriteComment(SectionComment.Pad);
			WriteSectionName(SectionName.Pad);

			for (int i = 0; i < padList.Count; i++)
			{
				string line = String.Format("{0}={1}", padList[i].Key, padList[i].Value);
				sw.WriteLine(line);
			}
		}

		private void WriteScreenSection(bool fullScreen, SyncTimer sync, int skip)
		{
			WriteComment(SectionComment.Screen);
			WriteSectionName(SectionName.Screen);

			sw.WriteLine(String.Format("FULL={0}",  fullScreen ? "1" : "0"));
			sw.WriteLine(String.Format("TIMER={0}", sync.Expression));
			sw.WriteLine(String.Format("SKIP={0:D}", skip));
		}

		private void WriteSeparator()
		{
			sw.WriteLine();
		}

		private void WriteComment(string title)
		{
			sw.WriteLine(String.Format("#{0}", title));
		}

		private void WriteSectionName(string name)
		{
			sw.WriteLine(String.Format("[{0}]", name));
		}

		private static class SectionComment
		{
			public static readonly string Key = "キーボード用のコンフィグ、キーをASCIIコードで記述（１０進数）";
			public static readonly string Pad = "パッド用のコンフィグ、ボタン番号を記述";
			public static readonly string Screen = "画面セッティング";
		}

		private static class SectionName
		{
			public static readonly string Key = "KEYCONFIG";
			public static readonly string Pad = "PADCONFIG";
			public static readonly string Screen = "SCREEN";
		}
	}
}